class Analytics {
    constructor(debug = false) {
        this.debug = debug;
    }

    async getOrCreateClientId() {
        let { clientId } = await chrome.storage.local.get('clientId');
        if (!clientId) {
            clientId = self.crypto.randomUUID();
            await chrome.storage.local.set({ clientId });
        }
        return clientId;
    }

    async getOrCreateSessionId() {
        let { sessionData } = await chrome.storage.session.get('sessionData');
        const currentTimeInMs = Date.now();
        if (sessionData && sessionData.timestamp) {
            const durationInMin = (currentTimeInMs - sessionData.timestamp) / 60000;
            if (durationInMin > GA_CONFIG.SESSION_EXPIRATION_IN_MIN) {
                sessionData = null;
            } else {
                sessionData.timestamp = currentTimeInMs;
                await chrome.storage.session.set({ sessionData });
            }
        }
        if (!sessionData) {
            sessionData = {
                session_id: currentTimeInMs.toString(),
                timestamp: currentTimeInMs.toString()
            };
            await chrome.storage.session.set({ sessionData });
        }
        return sessionData.session_id;
    }

    async fireEvent(name, params = {}) {
        if (!params.session_id) {
            params.session_id = await this.getOrCreateSessionId();
        }
        if (!params.engagement_time_msec) {
            params.engagement_time_msec = GA_CONFIG.DEFAULT_ENGAGEMENT_TIME_MSEC;
        }

        try {
            const response = await fetch(
                `${this.debug ? GA_CONFIG.GA_DEBUG_ENDPOINT : GA_CONFIG.GA_ENDPOINT
                }?measurement_id=${GA_CONFIG.MEASUREMENT_ID}&api_secret=${GA_CONFIG.API_SECRET}`,
                {
                    method: 'POST',
                    body: JSON.stringify({
                        client_id: await this.getOrCreateClientId(),
                        events: [
                            {
                                name,
                                params
                            }
                        ]
                    })
                }
            );
            if (!this.debug) {
                return;
            }
            console.log(await response.text());
        } catch (e) {
            console.error('Google Analytics request failed with an exception', e);
        }
    }
}

const GoogleAnalytics = new Analytics();